//
//  TodoListChooser.swift
//  Do It
//
//  Created by Jim Dovey on 1/27/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct Home: View {
    @EnvironmentObject private var data: DataCenter
    
    // START:NewList
    static private let listTemplate = TodoItemList(
        id: Int.min, name: "New List", color: .blue, icon: "list.bullet")
    @State private var showingEditor = false

    var body: some View {
        NavigationView {
            // <literal:elide> ... </literal:elide>
            // END:NewList
            // START:OnModify
            List {
                // START:NewSections
                Section(header: HomeHeader().padding(.vertical)) {
                    ForEach(data.todoLists) { list in
                        NavigationLink(destination: TodoList(list: list)) {
                            Row(name: list.name,
                                icon: list.icon,
                                color: list.color.uiColor)
                        }
                    }
                    // END:NewSections
                    // START_HIGHLIGHT
                    .onDelete { self.data.removeLists(atOffsets: $0) }
                    .onMove { self.data.moveLists(fromOffsets: $0, to: $1) }
                    // END_HIGHLIGHT
                    // START:NewSections
                }
                // END:NewSections
            }
            .font(.system(.headline, design: .rounded))
            .listStyle(GroupedListStyle())
            .navigationBarTitle("Lists")
            // START_HIGHLIGHT
            .navigationBarItems(leading: EditButton(), trailing: addButton)
            // END_HIGHLIGHT
            // END:OnModify
            // START:NewList
        }
        .sheet(isPresented: $showingEditor) {
            TodoListEditor(list: Self.listTemplate) //<label id="code.ch6.list.edit"/>
        }
    }
    
    private var addButton: some View {
        Button(action: { self.showingEditor.toggle() }) {
            Image(systemName: "plus.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Add New List"))
        }
    }
    // END:NewList

    private struct Row: View {
        var name: String
        var icon: String
        var color: Color

        var body: some View {
            HStack {
                Image(systemName: icon)
                    .foregroundColor(.white)
                    .frame(width: 32, height: 32)
                    .background(color)
                    .clipShape(Circle())
                Text(LocalizedStringKey(name))
            }
        }
    }
}

struct TodoListChooser_Previews: PreviewProvider {
    static var previews: some View {
        Home()
            .environmentObject(DataCenter())
            .padding(.top, 45)
    }
}
